/*
 * @(#)OMObject.java  1.0  2. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.oo.objectmodel;

import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import javax.swing.event.*;
import java.util.*;
import ch.hslu.cm.simulation.*;
/**
 * Simulates the structure and behavior of an object.
 * 
 *
 * @author  Werner Randelshofer
 * @version 1.0 2. Februar 2004  Created.
 */
public class OMObject extends AbstractElement implements SimulatedObjectListener {
    private String name;
    
    /**
     * This map holds the variables of the object.
     * The key is an instance of OMAttribute, the value is the value of the
     * variable.
     */
    private HashMap<OMAttribute, Object> variables = new HashMap<OMAttribute,Object>();
    
    private OMClass clazz;
    
    /** Creates a new instance. */
    public OMObject() {
        name = "objekt";
    }
    
    public OMClass getSimulatedClass() {
        return clazz;
    }
    public void setSimulatedClass(OMClass newValue) {
        OMClass oldValue = clazz;
        
        if (clazz != null) {
            clazz.removeSimulatedObjectListener(this);
        }
        clazz = newValue;
        if (clazz != null) {
            clazz.addSimulatedObjectListener(this);
        }
        firePropertyChange("simulatedClass", oldValue, newValue);
        fireClassChanged();
    }
    
    public void setName(String name) {
        if (! this.name.equals(name)) {
            this.name = name;
            fireNameChanged();
        }
    }
    public String getName() {
        return name;
    }
    
    public void set(String name, Object newValue) {
        set(getSimulatedClass(), name, newValue);
    }
    
    public Object get(String name) {
        return get(getSimulatedClass(), name);
    }
    public void set(OMClass definingClass, String name, Object newValue) {
        while (definingClass != null) {
            if (definingClass.isAttribute(name)) {
                set(definingClass.getAttribute(name), newValue);
                return;
            }
            definingClass = definingClass.getSuperclass();
        }
        throw new IllegalArgumentException("Variable "+name+" gibt es nicht.");
    }
    
    public Object get(OMClass clazz, String name) {
        while (clazz != null) {
            if (clazz.isAttribute(name)) {
                return get(clazz.getAttribute(name));
            }
            clazz = clazz.getSuperclass();
        }
        throw new IllegalArgumentException("Variable "+name+" gibt es nicht.");
    }
    public Object get(OMAttribute attribute) {
        if (variables.containsKey(attribute)) {
            return variables.get(attribute);
        } else {
            return attribute.getInitialValue();
        }
    }
    /**
     * Returns true, if the name denotes a variable name of this simulated object.
     */
    public boolean isVariable(OMClass clazz, String name) {
        while (clazz != null) {
            if (clazz.isAttribute(name)) {
                return true;
            }
            clazz = clazz.getSuperclass();
        }
        return false;
    }
    
    public void set(OMAttribute attribute, Object newValue) {
        System.out.println("SimulatedObject.set("+attribute+","+newValue+")");
        Object oldValue = variables.put(attribute, newValue);
        fireStateChanged("_"+attribute.getAttributeName(), oldValue, newValue);
    }
    
    public void addSimulatedObjectListener(OMObjectListener l) {
        listenerList.add(OMObjectListener.class, l);
    }
    
    public void removeSimulatedObjectListener(OMObjectListener l) {
        listenerList.remove(OMObjectListener.class, l);
    }
    /**
     * This is set to true, when the link is used to transmit a response.
     * This is set to false, when the link is used to transmit a message.
     */
    private boolean isResponse = false;
    
    /**
     * The message being transmitted by this link.
     */
    private Object message;
    /**
     * The response being transmitted by this link.
     */
    private Object response;
    
    /**
     * The receiver of the message.
     */
    private OMObject receiver;
    
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireStateChanged(String name, Object oldValue, Object newValue) {
        OMObjectEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==OMObjectListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new OMObjectEvent(this);
                ((OMObjectListener)listeners[i+1]).stateChanged(event);
            }
        }
    }
    
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireClassChanged() {
        OMObjectEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==OMObjectListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new OMObjectEvent(this);
                ((OMObjectListener)listeners[i+1]).classChanged(event);
            }
        }
    }
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireNameChanged() {
        OMObjectEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==OMObjectListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new OMObjectEvent(this);
                ((OMObjectListener)listeners[i+1]).nameChanged(event);
            }
        }
    }
    
    private void fireTransmittingMessage(OMLink link, String message) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, message, true, false);
                    ((OMObjectListener)listeners[i+1]).transmittingMessage(event);
                }
            }
        }
    }
    private void fireMessageTransmitted(OMLink link, Object message) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, message, false, false);
                    ((OMObjectListener)listeners[i+1]).messageTransmitted(event);
                }
            }
        }
    }
    private void fireTransmittingResponse(OMLink link, Object response, boolean isException) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, response, true, isException);
                    ((OMObjectListener)listeners[i+1]).transmittingResponse(event);
                }
            }
        }
    }
    private void fireResponseTransmitted(OMLink link, Object response, boolean isException) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, response, false, isException);
                    ((OMObjectListener)listeners[i+1]).responseTransmitted(event);
                }
            }
        }
    }
    
    public int getSimulatedConcept() {
        return ObjectModel.OBJECT;
    }
    
    public OMObject clone() {
        OMObject that = (OMObject) super.clone();
        that.variables = (HashMap<OMAttribute,Object>) this.variables.clone();
        return that;
    }
    
    public ObjectModel getObjectModel() {
        return (ObjectModel) getSimulation();
    }
    
    public void read(DOMInput in) throws IOException {
        setName(in.getAttribute("name", "objekt"));
        if (in.getElementCount("instanceof") > 0) {
            in.openElement("instanceof");
            setSimulatedClass((OMClass) in.readObject());
            in.closeElement();
        }
        
        int count = in.getElementCount("variable");
        for (int i=0; i < count; i++) {
            in.openElement("variable", i);
            if (in.getElementCount() == 2) {
                variables.put((OMAttribute) in.readObject(0), in.readObject(1));
            }
            in.closeElement();
        }
    }
    
    public void write(DOMOutput out) throws IOException {
        out.addAttribute("name", name);
        if (getSimulatedClass() != null) {
            out.openElement("instanceof");
            out.writeObject(getSimulatedClass());
            out.closeElement();
        }
        
        for (Iterator i=variables.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry entry = (Map.Entry) i.next();
            out.openElement("variable");
            out.writeObject((OMAttribute) entry.getKey());
            out.writeObject(entry.getValue());
            out.closeElement();
        }
    }
    
    public ArrayList<OMLink> getLinks(OMAssociation a) {
        ArrayList<OMLink> list = new ArrayList(getRelationships(ObjectModel.LINK));
        for (ListIterator i=list.listIterator(); i.hasNext(); ) {
            OMLink link = (OMLink) i.next();
            if (link.getAssociation() != a) {
                i.remove();
            }
        }
        return list;
    }
    
    public void objectAdded(SimulatedObjectEvent e) {
    }
    
    public void relationshipAdded(SimulatedObjectEvent e) {
    }
    
    public void relationshipRemoved(SimulatedObjectEvent e) {
    }
    
    public void objectChanged(SimulatedObjectEvent e) {
    }
    
    public void objectRequestRemove(SimulatedObjectEvent e) {
    }
    
    public void objectRemoved(SimulatedObjectEvent e) {
        if (e.getObject() == clazz) {
            setSimulatedClass(null);
            
        }
    }
}
